<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use CodeIgniter\HTTP\CLIRequest;
use CodeIgniter\HTTP\IncomingRequest;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Psr\Log\LoggerInterface;

class Articles extends BaseController
{
    protected $helpers = ['form', 'url', 'date'];
    protected $models = [];

    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);

        $this->db = \Config\Database::connect();
        $this->session = \Config\Services::session();
        $this->validation = \Config\Services::validation();
        $this->upload = \Config\Services::upload();

        $this->models = [
            'admin' => new \App\Models\AdminModel(),
            'category' => new \App\Models\CategoryModel(),
            'article' => new \App\Models\ArticleModel(),
            'author' => new \App\Models\AuthorModel(),
        ];

        date_default_timezone_set("Asia/Kolkata");
        error_reporting(0);
    }

    public function index()
    {
        $data['sidebar'] = "articles";
        $mAdminId = $this->session->get('login_admin_id');
        if ($mAdminId) {
            $data['articles'] = $this->models['article']->getAllParent();
            $data['authors'] = $this->models['author']->getAllParent();
            return view('admin/articles', $data);
        } else {
            return view('admin/signin', $data);
        }
    }

    public function addArticle()
    {
        $mAdminId = $this->session->get('login_admin_id');
        if ($mAdminId) {
            $data['sidebar'] = "articles";
            $data['authors'] = $this->models['author']->getAllParent();
            $data['categories'] = $this->models['category']->getAllParent();
            return view('admin/add_article', $data);
        } else {
            return view('admin/login');
        }
    }

    public function actionAddArticle()
    {
        $mUserId = $this->session->get('login_admin_id');
        if ($mUserId) {
            $data['active'] = "articles";
            if ($this->request->getMethod() === 'POST') {
                $rules = [
                    'category' => 'required',
                    'title' => 'required',
                    'description' => 'required',
                    'author' => 'required',
                    'date' => 'required',
                ];
    
                if (!$this->validate($rules)) {
                    return redirect()->to('admin/articles/addArticle')->with('error', 'Validation Error.')->withInput();
                }
    
                $mCategory = $this->request->getPost('category');
                $mTitle = $this->request->getPost('title');
                $mDate = $this->request->getPost('date');
                $mAuthor = $this->request->getPost('author');
                $mDesc = $this->request->getPost('description');
                $mDesc = nl2br(htmlentities($mDesc, ENT_QUOTES, 'UTF-8'));
                
                // Correct way to get uploaded files
                $pic = $this->request->getFile('image');
                $mToUploadProductPic = $this->articleFileUpload('image', $pic);
    
                $attach = $this->request->getFile('attachment');
                $mToUploadAttachment = $this->articleFileUpload('attachment', $attach);
    
                $mKey = $this->mGenerateRandomNumber();
                $data = [
                    'article_key' => $mKey,
                    'category_key' => $mCategory,
                    'article_author' => $mAuthor,
                    'article_date' => $mDate,
                    'article_title' => $mTitle,
                    'article_description' => $mDesc,
                    'article_image' => $mToUploadProductPic,
                    'article_attachment' => $mToUploadAttachment,
                    'article_date_added' => date('Y-m-d H:i:s'),
                    'article_date_updated' => date('Y-m-d H:i:s')
                ];
                $result = $this->models['article']->addParent($data);
                if ($result) {
                    return redirect()->to('admin/articles')->with('success', 'Article added.');
                } else {
                    return redirect()->to('admin/articles/addArticle')->with('error', 'Failed. Something went wrong.');
                }
            }
            // If not POST, load the form view
            return view('admin/add_article', $data);
        } else {
            return redirect()->to('admin/home');
        }
    }

    public function actionEditArticle($param = null)
    {
        $mUserId = $this->session->get('login_admin_id');
        if ($mUserId) {
            $data['active'] = "articles";
            $data['categories'] = $this->models['category']->getAllParent();
            $data['authors'] = $this->models['author']->getAllParent();
            if ($param) {
                $getCategory = $this->models['article']->getParentByKey($param);
                if (!empty($getCategory)) {
                    $data['article'] = $getCategory;
                    return view('admin/edit_article', $data);
                } else {
                    return redirect()->to('admin/articles')->with('error', 'Record not found.');
                }
            } else {
                return redirect()->to('admin/articles')->with('error', 'Something went wrong.');
            }
        } else {
            return view('admin/login');
        }
    }

    public function actionEnable($param = null)
    {
        $mUserId = $this->session->get('login_admin_id');
        if ($mUserId) {
            $data['active'] = "articles";
            if ($param) {
                $getCategory = $this->models['article']->getParentByKey($param);
                if (!empty($getCategory)) {
                    $data = [
                        'article_status' => 1,
                    ];
                    $result = $this->models['article']->updateParentByKey($param, $data);
                    if ($result) {
                        return redirect()->to('admin/articles')->with('success', 'Article updated.');
                    } else {
                        return redirect()->to('admin/articles')->with('error', 'Failed. Something went wrong.');
                    }
                } else {
                    return redirect()->to('admin/articles')->with('error', 'Record not found.');
                }
            } else {
                return redirect()->to('admin/articles')->with('error', 'Something went wrong.');
            }
        } else {
            return view('admin/login');
        }
    }

    public function actionDisable($param = null)
    {
        $mUserId = $this->session->get('login_admin_id');
        if ($mUserId) {
            $data['active'] = "articles";
            if ($param) {
                $getCategory = $this->models['article']->getParentByKey($param);
                if (!empty($getCategory)) {
                    $data = [
                        'article_status' => 0,
                    ];
                    $result = $this->models['article']->updateParentByKey($param, $data);
                    if ($result) {
                        return redirect()->to('admin/articles')->with('success', 'Article updated.');
                    } else {
                        return redirect()->to('admin/articles')->with('error', 'Failed. Something went wrong.');
                    }
                } else {
                    return redirect()->to('admin/articles')->with('error', 'Record not found.');
                }
            } else {
                return redirect()->to('admin/articles')->with('error', 'Something went wrong.');
            }
        } else {
            return view('admin/login');
        }
    }

    public function actionUpdateArticle($param = null)
    {
        $mUserId = $this->session->get('login_admin_id');
        if ($mUserId) {
            $data['active'] = "articles";
            if ($param) {
                $getCategory = $this->models['article']->getParentByKey($param);
                if (!empty($getCategory)) {
                    if ($this->request->getMethod() === 'POST') {
                        $rules = [
                            'category' => 'required',
                            'title' => 'required',
                            'description' => 'required',
                        ];

                        if (!$this->validate($rules)) {
                            return redirect()->to('admin/articles/actionEditArticle/' . $param)->with('error', 'Validation Error.');
                        }

                        $mCategory = $this->request->getPost('category');
                        $mTitle = $this->request->getPost('title');
                        $mDesc = $this->request->getPost('description');
                        $mDate = $this->request->getPost('date');
                        $mAuthor = $this->request->getPost('author');
                        $mDesc = nl2br(htmlentities($mDesc, ENT_QUOTES, 'UTF-8'));
                        $pic = $this->request->getPost('image');
                        $mToUploadProductPic = $this->articleFileUpload('image', $pic);
                        $changedImage = !empty($mToUploadProductPic) ? $mToUploadProductPic : $getCategory['article_image'];

                        $attach = $this->request->getPost('attachment');
                        $mToUploadAttachment = $this->articleFileUpload('attachment', $attach);
                        $changedAttach = !empty($mToUploadAttachment) ? $mToUploadAttachment : $getCategory['article_attachment'];

                        $data = [
                            'category_key' => $mCategory,
                            'article_title' => $mTitle,
                            'article_author' => $mAuthor,
                            'article_date' => $mDate,
                            'article_description' => $mDesc,
                            'article_image' => $changedImage,
                            'article_attachment' => $changedAttach,
                            'article_date_updated' => date('Y-m-d H:i:s')
                        ];
                        $result = $this->models['article']->updateParentByKey($param, $data);
                        if ($result) {
                            return redirect()->to('admin/articles/actionEditArticle/' . $param)->with('success', 'Article updated.');
                        } else {
                            return redirect()->to('admin/articles/actionEditArticle/' . $param)->with('error', 'Failed. Something went wrong.');
                        }
                    } else {
                        return redirect()->to('admin/articles/actionEditArticle/' . $param)->with('error', 'Something went wrong.');
                    }
                } else {
                    return redirect()->to('admin/articles')->with('error', 'Record not found.');
                }
            } else {
                return redirect()->to('admin/articles')->with('error', 'Something went wrong.');
            }
        } else {
            return view('admin/login');
        }
    }

    public function actionDeleteArticle($param)
    {
        $mUserId = $this->session->get('login_admin_id');
        if ($mUserId) {
            $data['active'] = "articles";
            if ($param) {
                $getCategory = $this->models['article']->getParentByKey($param);
                if (!empty($getCategory)) {
                    $deleteCategory = $this->models['article']->deleteParentByKey($param);
                    if ($deleteCategory) {
                        return redirect()->to('admin/articles')->with('success', 'Article deleted.');
                    } else {
                        return redirect()->to('admin/articles')->with('error', 'Failed, Something went wrong.');
                    }
                } else {
                    return redirect()->to('admin/articles')->with('error', 'Record not found.');
                }
            } else {
                return redirect()->to('admin/articles')->with('error', 'Something went wrong.');
            }
        } else {
            return view('admin/login');
        }
    }

    public function getSubcatByCat()
    {
        $categoryKey = $this->request->getPost('value');
        $subs = $this->models['subcategory']->getSubCatByCat($categoryKey);
        if (!empty($subs)) {
            $result = '<option disabled="" selected="">Select Sub-Category</option>';
            foreach ($subs as $sub) {
                $result .= "<option value='" . $sub['subcategory_key'] . "'>" . $sub['subcategory'] . "</option>" . PHP_EOL;
            }
            return $this->response->setJSON(['result' => $result]);
        } else {
            return $this->response->setJSON(['result' => "<option>No Sub Categories</option>"]);
        }
    }

    private function mGenerateRandomNumber(): string
    {
        return bin2hex(random_bytes(12));
    }

    private function articleFileUpload($mId, $mFile)
{
    if ($mFile->isValid() && !$mFile->hasMoved()) {
        $newName = $mFile->getRandomName();
        $mFile->move(FCPATH . 'uploads/articles', $newName);
        return $newName;
    }
    return '';
}
}